  ' -----------------------------------------------------------
  ' GY271 eCompass.bas
  ' A program for Micromite to communicate with the GY-271
  ' eCompass module (based on the Honeywell HMC5883L 3-axis sensor chip)
  ' as a digital electronic compass
  ' 
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 10/8/2018 at 1:15 pm
  '
  ' Notes:
  ' 1. Communication with the HMC5883L chip is via the Micromite's
  ' I2C port, which uses pin 17 for SCL and pin 18 for SDA.
  ' 2. Although the HMC5883L chip's I2C address is specified as 3Dh for
  ' reading and 3Ch for writing, MMBasic's I2C function expects a
  ' 7-bit I2C address since it adds the lsb read/write bit automatically.
  ' So in this program we use 1Eh as the chip's I2C address.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER Xval = 0       ' X axis measurement (16 bits -> +/-12 bits)
  DIM AS INTEGER Yval = 0       ' Y axis measurement (16 bits -> +/-12 bits)
  DIM AS INTEGER Zval = 0       ' Z axis measurement (16 bits -> +/-12 bits)
  DIM AS INTEGER StReg = 0      ' variable to save status register
  DIM INTEGER HByte(6)          ' one-dim array to save measurement data bytes
  
  DIM AS FLOAT Declin! = 12.583 ' replace this value with your own declination
                                ' in degrees (+ for EAST, - for WEST)
  DIM AS FLOAT Heading!         ' FP value for current magn heading in degrees
  DIM AS FLOAT TruHdng!         ' FP value for calc true heading in degrees

  DIM AS STRING MHdng$          ' string for display of magnetic heading
  DIM AS STRING THdng$          ' string for display of true heading
  DIM AS STRING Decl$           ' string for display of mag declination
  DIM AS STRING EWStr$          ' string for East or West indication
  
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  PIN(17) = 0             ' set pin 17 to low and
  SETPIN 17, DOUT         ' declare it a digital output (SCL)
  I2C OPEN 100, 200       ' enable the Micromite I2C port in master mode
  ' ----------------------------------------------------------------------
  ' first show the opening screen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes/16, "SILICON CHIP", CM, 1, 2, Green, Black
  TEXT MM.HRes/2, MM.VRes*3/16, "eCOMPASS (GY-271)",CM, 1, 2, Green, Black
  
  ' now initialise the HMC5883L for simple compass use
    I2C WRITE &H1E,0,2,&H00,&H70  ' set CRA for 8X avg,15Hz, normal measurement
    I2C WRITE &H1E,0,2,&H01,&HA0  ' set CRB for Gain = 390
    I2C WRITE &H1E,0,2,&H02,&H00  ' set Mode reg for continuous measurement
  PRINT "HMC5883L device Initialised"
  PRINT " "
  
  ' get string to represent declared magnetic declination
  Decl$ = STR$(Declin!, 3, 3) 
  
  ' main program loop starts here
  DO
    GetHeading          ' now get the current heading
    DispRes             ' and display the results on screen
    PAUSE 500           ' set to loop about once every 500ms
  LOOP
  I2C CLOSE
END ' end of main part of program, subroutines follow
  
  ' *****************************************************************
  ' subroutine to get current heading info from HMC5883L
SUB GetHeading
  I2C WRITE &H1E,0,1, &H09  ' first do a dummy write to status reg
                            ' to set the address pointer to this reg
  DO      ' then read the register
    I2C READ &H1E,0,1,StReg
    IF (StReg AND &H01) = 1 THEN EXIT DO  ' if RDY bit set, proceed
  LOOP                        ' otherwise keep looping
  
  I2C WRITE &H1E,0,1,&H03    ' set register address ptr to DXRA0
  PAUSE 80                   ' allow time for measurements      
  I2C READ &H1E,0,6, HByte() ' fetch the 6 measurement bytes, then
  ' glue each pair together to get the 3 axis readings (note the odd
  ' order: X,Z then Y) & mask back to the least significant 12 bits
  Xval = ((HByte(0) << 8) OR HByte(1)) AND &H0000000000000FFF
  Zval = ((HByte(2) << 8) OR HByte(3)) AND &H0000000000000FFF
  Yval = ((HByte(4) << 8) OR HByte(5)) AND &H0000000000000FFF
  
  ' now convert to 2's complement if values exceed 2047
  IF Xval > 2047 THEN Xval = Xval - 4096 
  IF Yval > 2047 THEN Yval = Yval - 4096
  
  ' now work out heading in radians (with respect to magn North)
  SELECT CASE Yval
  CASE 0
    IF Xval < 0 THEN Heading! = PI/2 ELSE Heading! = -PI/2
  CASE IS > 0         ' i.e., when Yval is positive
    Heading! = -ATN(Xval/Yval)
  CASE IS < 0         ' but if Yval is negative
    IF Xval < 0 THEN  ' if Xval is negative
      Heading! = PI - ATN(Xval/Yval)
    ELSE              ' but if Xval is positive
      Heading! = -PI - ATN(Xval/Yval)
    END IF
  END SELECT
   
  ' and then convert to degrees
  Heading! = Heading! * 180/PI
  MHdng$ = STR$(Heading!, 4, 3)   
  ' and finally correct for magnetic declination
  TruHdng! = Heading! - Declin!
  THdng$ = STR$(TruHdng!, 4, 3)
  IF TruHdng! > 0 THEN EWStr$ = " deg (EAST)" ELSE EWStr$ = " deg (WEST)"
  PRINT "Xval = ",Xval,"  Yval = ",Yval, " Hdg = ", Heading!, " degrees"
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display heading on screen
SUB DispRes
  TEXT MM.HRes/2, MM.VRes*3/8, "Magnetic Heading:", CM, 1, 2, White, Black
  TEXT MM.HRes/2, MM.VRes/2, MHdng$ + " degrees", CM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes*5/8, "True Heading:", CM, 1, 2, White, Black
  TEXT MM.HRes/2, MM.VRes*3/4, THdng$ + EWStr$, CM, 1, 2, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes*7/8, "(Declination = " + Decl$ + " degrees)", CM, 1, 1, White, Black
END SUB
  ' *******************************************************************
